<?php

/**
 * Plugin Name: WooCommerce Related Products C4W
 * Description: Dynamic rules for Upsell and Cross Sell. Define the rules for related products and offer people alternatives to the current product automatically and only those you need.
 * Version: 2.1.4
 * Author: Custom4Web
 * Author URI: https://www.custom4web.com/
 * Text Domain: wc-alp
 * Domain Path: /languages/
 * WC requires at least: 3.0.0
 * WC tested up to: 4.0.1
 */
if ( !defined( 'ABSPATH' ) ) {
    exit; // Exit if accessed directly.
}

if ( !class_exists( 'WC_DRP' ) ) :

    /**
     * Main Class.
     */
    final class WC_DRP {

        const INCLUDE_PATH = 'includes';
        const ADMIN_PATH   = 'admin';
        const VERSION      = '2.1.4';
        const PREFIX       = 'wcdrp_';

        /**
         * The single instance of the class.
         */
        protected static $_instance = null;
        protected $loader;
        protected $post_type;

        /**
         * Main Instance.
         *
         * Ensures only one instance of WooCommerce is loaded or can be loaded.
         */
        public static function instance() {
            if ( is_null( self::$_instance ) ) {
                self::$_instance = new self();
            }

            return self::$_instance;
        }

        /**
         * Cloning is forbidden.
         */
        public function __clone() {
            _doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'wc-alp' ), '2.1' );
        }

        /**
         * Unserializing instances of this class is forbidden.
         */
        public function __wakeup() {
            _doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'wc-alp' ), '2.1' );
        }

        /**
         * WooCommerce Constructor.
         */
        public function __construct() {
            $this->add_hooks();
            $this->includes();
            $this->load();
            $this->init();
            add_action( 'plugins_loaded', array( $this, 'load_plugin_textdomain' ) );
        }

        /**
         * Include required core files used in admin and on the frontend.
         */
        public function load() {
            $this->loader = new Custom4Web_Wp_Autoloader( $this->get_path() );
            if ( $this->is_request( 'admin' ) ) {
                $this->loader->autoload( self::ADMIN_PATH );
            }
            $this->loader->autoload( self::INCLUDE_PATH );
        }

        public function init() {
            global $WCDRP_License;
            $WCDRP_License = new WCDRP_License();
            new WCDRP_No_Conflict();
            $this->post_type = new WCDRP_Post_type();
            if ( $this->is_request( 'admin' ) ) {
                new WCDRP_Admin( $this->post_type );
            }
            add_action( 'after_switch_theme', array( $this, 'activation' ) );            
            add_action( 'woocommerce_settings_saved', array( $this, 'check_license' ) );
            add_filter( 'plugin_action_links_' . plugin_basename(  __FILE__  ), array( $this, 'wc_alp_plugin_action_links' ) );
        }
        
        public function wc_alp_plugin_action_links( $links ) {
            $mylinks = array(
                '<a href="' . admin_url( 'admin.php?page=wc-settings&tab=products&section=wcdrp' ) . '">' . __( 'License', 'wc-alp' ) . '</a>',
                '<a href="' . admin_url( 'edit.php?post_type=wcdrp_rules' ) . '">' . __( 'Rules', 'wc-alp' ) . '</a>',
            );
            return array_merge( $links, $mylinks );
        }

        public function includes() {
            include_once( 'vendor/custom4web/custom4web-wp-autoloader.php' );
            include_once( 'vendor/custom4web/custom4web-wp-template.php' );
            include_once( 'vendor/custom4web/custom4web-wp-cpt.php' );
        }

        public function add_hooks() {
            add_action( 'wp_enqueue_scripts', array( $this, 'load_resources' ) );
            add_action( 'admin_enqueue_scripts', array( $this, 'load_resources' ) );
        }

        public function activation() {
            update_option( self::PREFIX . 'hooks', array() );
        }

        public function check_license() {            
            if ( !empty( $_POST['wcdrp_license_name'] ) && !empty( $_POST['wcdrp_license_code'] ) && empty( $_POST['linked_deactivation'] ) ) {
                global $WCDRP_License;
                $license_code = $_POST['wcdrp_license_code'];
                $license_name = $_POST['wcdrp_license_name'];
                $result       = $WCDRP_License->verify_license( false, $license_code, $license_name );
                if ( !$result['status'] ) {
                    $result       = $WCDRP_License->activate_license( $license_code, $license_name );
                }
                if ( empty( $result['status'] ) ) {
                   WC_Admin_Settings::add_error( __( 'The user name or a license code is wrong.', 'wc-alp' ) );
                } else {
                   WC_Admin_Settings::add_message( $result['message'] );
                   update_option( 'wcdrp_license_true', $result );
                }
            }elseif( !empty( $_POST['linked_deactivation'] )){
                global $WCDRP_License;
                $license_code = $_POST['wcdrp_license_code'];
                $license_name = $_POST['wcdrp_license_name'];                
                $result       = $WCDRP_License->deactivate_license( $license_code, $license_name ); 
                if ( empty( $result['status'] ) ) {
                   WC_Admin_Settings::add_error( $result['message'] );
                } else {
                   WC_Admin_Settings::add_message( $result['message'] );
                   delete_option( 'wcdrp_license_true' );
                }
            }           
        }

        /**
         * What type of request is this?
         *
         * @param  string $type admin, ajax, cron or frontend.
         *
         * @return bool
         */
        public function is_request( $type ) {
            switch ( $type ) {
                case 'admin' :
                    return is_admin();
                case 'ajax' :
                    return defined( 'DOING_AJAX' );
                case 'frontend' :
                    return (!is_admin() || defined( 'DOING_AJAX' ) );
            }

            return false;
        }

        /**
         * Load localization files.
         */
        public function load_plugin_textdomain() {
            load_plugin_textdomain( 'wc-alp', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );
        }

        /**
         * Get the plugin url.
         * @return string
         */
        public function get_url() {
            return untrailingslashit( plugins_url( '/', __FILE__ ) );
        }

        /**
         * Get the plugin path.
         * @return string
         */
        public function get_path() {
            return untrailingslashit( plugin_dir_path( __FILE__ ) );
        }

        /**
         * Get the template path.
         * @return string
         */
        public function get_template_path() {
            return apply_filters( 'woocommerce_template_path', 'wc_related_products/' );
        }

        /**
         * Get Ajax URL.
         * @return string
         */
        public function ajax_url() {
            return admin_url( 'admin-ajax.php', 'relative' );
        }

        public static function load_resources() {
            $plugin_js    = apply_filters( 'linked_products_js', true );
            $plugin_css   = apply_filters( 'linked_products_css', true );
            $carousel_js  = apply_filters( 'linked_products_carousel_js', true );
            $carousel_css = apply_filters( 'linked_products_carousel_css', true );

            wp_register_script(
                    self::PREFIX . 'wcdrp-carousel', plugins_url( 'assets/js/owl.carousel.js', __FILE__ ), array( 'jquery' ), self::VERSION, true
            );

            wp_register_script(
                    self::PREFIX . 'wc-auto-linked-products', plugins_url( 'assets/js/wc-auto-linked-products.js', __FILE__ ), array( 'jquery' ), self::VERSION, true
            );

            wp_register_style(
                    self::PREFIX . 'admin-style', plugins_url( 'assets/css/admin.css', __FILE__ ), array(), self::VERSION, 'all'
            );

            wp_register_style(
                    self::PREFIX . 'admin-style-chosen', plugins_url( 'assets/css/chosen.css', __FILE__ ), array(), self::VERSION, 'all'
            );

            wp_register_style(
                    self::PREFIX . 'wc-auto-linked-products', plugins_url( 'assets/css/style.css', __FILE__ ), array(), self::VERSION, 'all'
            );

            wp_register_style(
                    self::PREFIX . 'wcdrp-carousel', plugins_url( 'assets/css/owl.carousel.css', __FILE__ ), array(), self::VERSION, 'all'
            );

            wp_register_style(
                    self::PREFIX . 'wcdrp-carousel-default', plugins_url( 'assets/css/owl.theme.default.css', __FILE__ ), array(), self::VERSION, 'all'
            );

            if ( is_admin() ) {
                wp_enqueue_style( self::PREFIX . 'admin-style' );
                wp_enqueue_style( self::PREFIX . 'admin-style-chosen' );


                wp_enqueue_script( self::PREFIX . 'repeater', plugins_url( 'assets/js/jquery.repeater.js', __FILE__ ), array( 'jquery' ), self::VERSION, true
                );
                wp_enqueue_script( self::PREFIX . 'chosen', plugins_url( 'assets/js/chosen.jquery.js', __FILE__ ), array( 'jquery' ), self::VERSION, true
                );
                wp_localize_script( self::PREFIX . 'chosen', 'wcdrp', array(
                    'ajax_url' => admin_url( 'admin-ajax.php' ),
                    'action'   => 'wcdrp_additional_settings'
                ) );

                wp_enqueue_script( self::PREFIX . 'updater', plugins_url( 'assets/js/wc-alp-admin.js', __FILE__ ), array( 'jquery' ), self::VERSION, true
                );
                wp_localize_script( self::PREFIX . 'updater', 'wcdrp_updater', array(
                    'ajaxUrl'         => admin_url( 'admin-ajax.php' ),
                    'action'          => 'update_wcdrp'
                ) );
            } else {
                if ( $plugin_css ) {
                    wp_enqueue_style( self::PREFIX . 'wc-auto-linked-products' );
                }
                if ( $carousel_css ) {
                    wp_enqueue_style( self::PREFIX . 'wcdrp-carousel' );
                    wp_enqueue_style( self::PREFIX . 'wcdrp-carousel-default' );
                }
                if ( $carousel_js ) {
                    wp_enqueue_script( self::PREFIX . 'wcdrp-carousel' );
                }
                if ( $plugin_js ) {
                    wp_enqueue_script( self::PREFIX . 'wc-auto-linked-products' );
                }
            }
        }

        public function get_wc_filter_data( $match ) {
            global $wp_filter;

            $prefix      = 'woocommerce';
            $filter_data = array();

            foreach ( $wp_filter as $tag => $priorities ) {
                if ( false !== strpos( $tag, $prefix ) ) {
                    foreach ( $priorities as $priority => $hooks ) {
                        $hook = current( preg_grep( "/_$match$/i", array_keys( $hooks ) ) );
                        if ( $hook ) {
                            $filter_data = array( $tag, $hook, $priority );
                        }
                    }
                }
            }

            return $filter_data;
        }

    }

    endif;

/**
 * Main instance of Linked Products.
 *
 * Returns the main instance of WC_Related_Products to prevent the need to use globals.
 *
 * @return WC_DRP
 */
function WC_DRP() {
    return WC_DRP::instance();
}

if ( in_array( 'woocommerce/woocommerce.php', apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) ) ) {
    WC_DRP();
    register_activation_hook( __FILE__, array( WC_DRP(), 'activation' ) );
}