<?php

/**
 * Define Constants
 * ================
 */
define( 'BLOGMAN_CONFIG_ID', 'blogman' );
define( 'BLOGMAN_THEME_PATH', get_template_directory() );
define( 'BLOGMAN_THEME_URL', esc_url( get_template_directory_uri() ) );
/**
 * Global content width
 *
 * @param $content_width
 *
 * @since 1.0
 * @return void
 */
if ( ! isset( $content_width ) )
	$content_width = 1140;

/**
 * Theme setup
 * Hook to action after_setup_theme
 *
 * @since 1.0
 */
if ( ! function_exists( 'blogman_theme_setup' ) ) {
	function blogman_theme_setup() {
		
		load_theme_textdomain( 'blogman', get_template_directory() . '/languages' );
		
		// Feed Links
		add_theme_support( 'automatic-feed-links' );
		
		// Title tag
		add_theme_support( 'title-tag' );
		
		// Post Formats
		add_theme_support( 'post-formats', array( 'link', 'video', 'audio' ) );
		
		// Default custom header.
		add_theme_support( 'custom-header' );
		
		// Default custom backgrounds.
		add_theme_support( 'custom-background' );
		
		// Post thumbnails
		add_theme_support( 'post-thumbnails' );
		
		add_theme_support( 'align-wide' );
		
		// Register navigation menu
		register_nav_menus( array(
			'main-menu'   => 'Primary Menu'
		) );
		
		// Adds the support for the Custom Logo introduced in WordPress 4.5
		add_theme_support( 'custom-logo',
			array(
					'height'      => 56,
					'width'       => 224,
					'flex-height' => true,
					'flex-width'  => true,
			)
		);
	}
}
add_action( 'after_setup_theme', 'blogman_theme_setup' );

/**
 * Register widget area.
 */
if ( ! function_exists( 'blogman_widgets_init' ) ) {
	function blogman_widgets_init() {
		
		register_sidebar( array(
			'name'          => esc_html__( 'Main Sidebar', 'blogman' ),
			'id'            => 'main-sidebar',
			'description'   => esc_html__( 'Add widgets here to appear in your sidebar.', 'blogman' ),
			'before_widget' => '<aside id="%1$s" class="widget %2$s">',
			'after_widget'  => '</aside>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		
		register_sidebar( array(
			'name'          => esc_html__( 'Footer', 'blogman' ),
			'id'            => 'footer-widget',
			'description'   => esc_html__( 'Appears in the site footer.', 'blogman' ),
			'before_widget' => '<aside id="%1$s" class="widget %2$s">',
			'after_widget'  => '</aside>',
			'before_title'  => '<h3 class="widget-title">',
			'after_title'   => '</h3>',
		) );
		
		register_sidebar( array(
			'name'          => esc_html__( 'Footer Instagram', 'blogman' ),
			'id'            => 'footer-instagram',
			'before_widget' => '<div id="%1$s" class="instagram-feeds-container %2$s">',
			'after_widget'  => '</div>',
			'before_title'  => '<div class="instagram-title"><h4 class="footer-instagram-title"><a><i class="fab fa-instagram"></i>',
			'after_title'   => '</a></h4></div>',
			'description'   => esc_html__( 'Only use for Instagram feed. Display instagram images on your website footer. Required plugin: https://wordpress.org/plugins/instagram-feed/. Simply add a Custom HTML Widget by dragging it into this box, then write "[instagram-feed num=8 cols=8 showheader=false  showbio=false imagepadding=0 imagepaddingunit=px showbutton=false showfollow=false]" code into it.', 'blogman' ),
		) );
		
	}
}
add_action( 'widgets_init', 'blogman_widgets_init' );

/**
 * Enqueue styles
 */
if ( ! function_exists( 'blogman_load_styles' ) ) {
	function blogman_load_styles() {
		
		// Bootstrap css
		wp_enqueue_style( 'bootstrap', BLOGMAN_THEME_URL .'/css/bootstrap.min.css', false, null );
		wp_enqueue_style( 'bootsnav', BLOGMAN_THEME_URL .'/css/bootsnav.css', false, null );
		// Font Awesome icon
		wp_enqueue_style( 'font-awesome', BLOGMAN_THEME_URL .'/css/fontawesome-all.css', false, null );
		// Load our main stylesheet.
		wp_enqueue_style( 'blogman-style', BLOGMAN_THEME_URL . '/style.css',false, null );
		
		
		// js for comments
		if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
			wp_enqueue_script( 'comment-reply' );
		}
	}
}
add_action( 'wp_enqueue_scripts', 'blogman_load_styles' );

/**
 * Registers an editor stylesheet for the theme.
 */
function blogman_editor_styles() {
    add_editor_style( array('editor-style.css','css/fontawesome-all.css') );
}
add_action( 'admin_init', 'blogman_editor_styles' );

/**
 * Enqueue scripts
 */
if ( ! function_exists( 'blogman_load_scripts' ) ) {
	function blogman_load_scripts() {
		
		wp_enqueue_script( 'bootstrap', BLOGMAN_THEME_URL . '/js/bootstrap.min.js', array( 'jquery' ), '', true );
		wp_enqueue_script( 'bootsnav', BLOGMAN_THEME_URL . '/js/bootsnav.js', array( 'jquery' ), '', true );
		if ( true == get_theme_mod( 'smooth_scrolling', false ) ) {
		wp_enqueue_script( 'SmoothScroll', BLOGMAN_THEME_URL . '/js/SmoothScroll.js', array( 'jquery' ), '', true );
		}
		wp_enqueue_script( 'blogman-js', BLOGMAN_THEME_URL . '/js/function.js', array( 'jquery' ), '', true );
		
		wp_enqueue_script( 'html5shiv', BLOGMAN_THEME_URL . '/js/html5shiv.min.js','', null);
		wp_script_add_data( 'html5shiv', 'conditional', 'lt IE 9' );

		wp_enqueue_script( 'respond', BLOGMAN_THEME_URL . '/js/respond.min.js','', null );
		wp_script_add_data( 'respond', 'conditional', 'lt IE 9' );
    
	}
}
add_action( 'wp_enqueue_scripts', 'blogman_load_scripts' );


/**
 * Add a pingback url auto-discovery header for singularly identifiable articles.
 */
function blogman_pingback_header() {
	if ( is_singular() && pings_open() ) {
		printf( '<link rel="pingback" href="%s">' . "\n", get_bloginfo( 'pingback_url' ) );
	}
}
add_action( 'wp_head', 'blogman_pingback_header' );

/**
 * Allowed HTM for wp_kses
 */

if ( ! function_exists( 'blogman_wp_kses_allowed_html' ) ) {
	function blogman_wp_kses_allowed_html() {
		
		$allowed_tags = wp_kses_allowed_html( 'post' );
		$allowed_tags['iframe'] = array(
			'src'=> true,
			'width'=> true,
			'height'=> true,
			'frameborder'=> true,
			'allowfullscreen'=> true,
			'webkitallowfullscreen'=> true,
			'mozallowfullscreen'=> true,
			'scrolling'=> true,
		);
		return $allowed_tags;
	}
}

/**
 * Displays social links 
 */
if ( ! function_exists( 'blogman_social_links' ) ) {
	function blogman_social_links( $text = true ) {
		$sociallinks = get_theme_mod( 'social_links_items'); 
		$output = '';
		if( is_array( $sociallinks ) && count( $sociallinks ) > 0 ) {
			
			foreach($sociallinks as $sitem) {
				$output .= '<a target="_blank" href="'.esc_url( $sitem['social_link'] ).'"><i class="fab fa-'.esc_attr( $sitem['social_icon'] ).'"></i></a>';
			}
		}
		return $output;
	}
}

/**
 * Toolkit for Customizer 
 */

require_once BLOGMAN_THEME_PATH . '/inc/include-kirki.php';
require_once BLOGMAN_THEME_PATH . '/inc/class-blogman-kirki.php';
require_once BLOGMAN_THEME_PATH . '/inc/customizer/customizer.php';

/**
 * Rcommend plugins notice
 */
require_once BLOGMAN_THEME_PATH . '/inc/required-plugins.php';


/**
 * Custom the_excerpt() length function
 */
if ( ! function_exists( 'blogman_custom_excerpt_length' ) ) {
	function blogman_custom_excerpt_length( $length ) {
		$number_excerpt_length = get_theme_mod( 'excerpt_length_post',25 );
		return $number_excerpt_length;
	}
}
add_filter( 'excerpt_length', 'blogman_custom_excerpt_length', 999 );

/**
 * Custom the_excerpt() more string
 */
if ( ! function_exists( 'blogman_new_excerpt_more' ) ) {
	function blogman_new_excerpt_more( $more ) {
		if( true == get_theme_mod( 'excerpt_read_more_sign',true ) ):
			return ' &hellip;';
		else:
			return '';
		endif;
	}
}
add_filter( 'excerpt_more', 'blogman_new_excerpt_more' );

/**
 * Custom comment walker
 */
function blogman_comment_template( $comment, $args, $depth ) {
	?>
	<?php $add_below = ''; ?>
	<li <?php comment_class(); ?> id="comment-<?php comment_ID() ?>">
		<div class="the-comment">
			<div class="comment-header">
				<div class="avatar"><?php echo get_avatar( $comment, 78 ); ?></div>
				<div class="author-meta">
					<h5>
						<?php echo get_comment_author_link(); ?>
						<?php comment_reply_link( array_merge( $args, array( 'reply_text' => __( 'Reply', 'blogman' ), 'add_below' => 'comment', 'depth' => $depth, 'max_depth' => $args['max_depth'] ) ) ); ?>
					</h5>
						
					<span class="comment-date"><?php printf( esc_attr__( '%1$s at %2$s', 'blogman' ), get_comment_date(),  get_comment_time() ); ?><?php edit_comment_link( __( ' - Edit', 'blogman' ),'  ','' ); ?></span>
					
				</div>
			</div>
			
			<div class="comment-body">
				<?php if ( '0' == $comment->comment_approved ) : ?>
					<em><?php esc_attr_e( 'Your comment is awaiting moderation.', 'blogman' ); ?></em>
					<br />
				<?php endif; ?>
				<?php comment_text() ?>
			</div>
		</div>
	<?php
}

/**
 * Add code in <head> section
 */
function blogman_custom_code_in_head(){
	echo get_theme_mod( 'custom_code_in_head', '' );
}
add_action('wp_head', 'blogman_custom_code_in_head');

/**
 * Add code in footer section
 */
function blogman_custom_code_in_footer(){
	echo get_theme_mod( 'custom_code_in_footer', '' );
}
add_action('wp_footer', 'blogman_custom_code_in_footer', 100);


/**
  *	Include the alternative webfonts loading method.
  * @since 1.1
  */
$file_path = wp_normalize_path( get_template_directory() . '/inc/class-kirki-modules-webfonts-link.php' );
if ( file_exists( $file_path ) && ! class_exists( 'Kirki_Modules_Webfonts_Link' ) ) {
	include_once $file_path;
}
if ( ! function_exists( 'blogman_change_fonts_load_method' ) ) {
	/**
	 * Changes the font-loading method.
	 *
	 * @param string $method The font-loading method (async|link)
	 */
	function blogman_change_fonts_load_method( $method ) {
		// Check for a theme-mod.
		// We don't want to force the use of the link method for googlefonts loading
	    // since the async method is better in general.
		if ( 'link' === get_theme_mod( 'webfonts_load_mothod','link' ) ) {
			return 'link';
		}
		return $method;
	}
}
add_filter( 'kirki_googlefonts_load_method', 'blogman_change_fonts_load_method' );


/**
 * Social share links
 * @since 1.1
 */
if ( ! function_exists( 'blogman_social_share_links' ) ) {
	function blogman_social_share_links() {
		
		$social_sharing_links = get_theme_mod( 'social_sharing_links', array(
				'facebook', 'twitter', 'linkedin', 'googleplus', 'pinterest', 'email',
				)
			);
		
		$social_sharing_title = get_theme_mod( 'social_sharing_title');
		
		if( empty($social_sharing_title) ):
			$social_sharing_title = __( 'Share on :', 'blogman' );
		endif;
		
		if( empty($social_sharing_links) )
			return;
		
		$social_links_array = array();
		$link		= get_the_permalink();
		$title 		= get_the_title();
		$content	= get_the_content();
		$content 	= strip_shortcodes($content);
		$content 	= strip_tags($content);
		$image 		= '';
		
		if( has_post_thumbnail( get_the_ID() ) ) :
			$image = wp_get_attachment_url( get_post_thumbnail_id( get_the_ID() ) );
		endif;
		
		$social_links_array['facebook'] = array(
				'url'        => 'https://www.facebook.com/sharer.php?u=' . rawurlencode( $link ) . '&t=' . rawurlencode( $title ),
				'icon'		 => 'fab fa-facebook-f',
			);
		
		$social_links_array['twitter'] = array(
				'url'        => 'https://twitter.com/share?text=' . rawurlencode( html_entity_decode( $title, ENT_COMPAT, 'UTF-8' ) ) . '&url=' . rawurlencode( $link ),
				'icon'		 => 'fab fa-twitter',
			);
		
		$social_links_array['linkedin'] = array(
				'url'        => 'https://www.linkedin.com/shareArticle?mini=true&url=' . $link . '&amp;title=' . rawurlencode( $title ) . '&amp;summary=' . rawurlencode( mb_substr( html_entity_decode( $content, ENT_QUOTES, 'UTF-8' ), 0, 256 ) ),
				'icon'		 => 'fab fa-linkedin-in',
			);
			
		$social_links_array['reddit'] = array(
				'url'        => 'https://reddit.com/submit?url=' . $link . '&amp;title=' . rawurlencode( $title ),
				'icon'		 => 'fab fa-reddit-alien',
			);
		
		$social_links_array['tumblr'] = array(
				'url' => 'https://www.tumblr.com/share/link?url=' . rawurlencode( $link ) . '&amp;name=' . rawurlencode( $title ) . '&amp;description=' . rawurlencode( $content ),
				'icon'		 => 'fab fa-tumblr',
			);
		
		$social_links_array['googleplus'] = array(
				'url'        => 'https://plus.google.com/share?url=' . $link,
				'icon'		 => 'fab fa-google-plus-g',
			);
		
		$social_links_array['pinterest'] = array(
				'url' => 'https://pinterest.com/pin/create/button/?url=' . rawurlencode( $link ) . '&amp;description=' . rawurlencode( $content ) . '&amp;media=' . rawurlencode( $image ),
				'icon'		 => 'fab fa-pinterest-p',
			);
		
		$social_links_array['vk'] = array(
				'url'        => 'https://vkontakte.ru/share.php?url=' . rawurlencode( $link ) . '&amp;title=' . rawurlencode( $title ) . '&amp;description=' . rawurlencode( $content ),
				'icon'		 => 'fab fa-vk',
			);
		
		$social_links_array['email'] = array(
				'url'        => 'mailto:?subject=' . rawurlencode( $title ) . '&body=' . $link,
				'icon'		 => 'fas fa-envelope',
			);
		
		$social_links_array['whatsapp'] = array(
				'url'        => 'https://api.whatsapp.com/send?text=' . rawurlencode( $link ),
				'icon'		 => 'fab fa-whatsapp',
			);
		
		$social_links_array['stumbleupon'] = array(
				'url'        => 'https://www.stumbleupon.com/submit?url=' . rawurlencode( $link ) . '&amp;title=' . rawurlencode( $title ),
				'icon'		 => 'fab fa-stumbleupon',
			);

		$social_links_array['telegram'] = array(
				'url'        => 'https://telegram.me/share/url?url=' . rawurlencode( $link ) . '&amp;text=' . rawurlencode( $title ),
				'icon'		 => 'fab fa-telegram-plane',
			);
			
		?>
		<div class="social-share-post">
		<span><?php echo esc_html( $social_sharing_title ); ?> </span>
		<?php
		foreach($social_sharing_links as $site) :
			$url = $social_links_array[$site]['url'];
			$icon = $social_links_array[$site]['icon'];
		?>
			<a rel="nofollow" target="_blank" href="<?php echo esc_url( $url ); ?>"><i class="<?php echo esc_attr( $icon ); ?>"></i></a>
		<?php
		endforeach;
		echo '</div>';
	}
}

/*
 * Custom post format admin UI
 */
if( function_exists('acf_add_local_field_group') ):

acf_add_local_field_group(array(
	'key' => 'group_5c0df911d81e7',
	'title' => __('Enter Audio Detail', 'blogman'),
	'fields' => array(
		array(
			'key' => 'field_5c0df911dfb00',
			'label' => __('AUDIO URL (OEMBED) OR EMBED CODE', 'blogman'),
			'name' => '_format_audio_embed',
			'type' => 'textarea',
			'instructions' => '',
			'required' => 0,
			'conditional_logic' => 0,
			'wrapper' => array(
				'width' => '',
				'class' => '',
				'id' => '',
			),
			'default_value' => '',
			'placeholder' => '',
			'maxlength' => '',
			'rows' => 4,
			'new_lines' => '',
		),
	),
	'location' => array(
		array(
			array(
				'param' => 'post_format',
				'operator' => '==',
				'value' => 'audio',
			),
		),
	),
	'menu_order' => 0,
	'position' => 'normal',
	'style' => 'default',
	'label_placement' => 'top',
	'instruction_placement' => 'label',
	'hide_on_screen' => '',
	'active' => 1,
	'description' => '',
));

acf_add_local_field_group(array(
	'key' => 'group_5c0df9805e5ef',
	'title' => __('Enter Link Detail', 'blogman'),
	'fields' => array(
		array(
			'key' => 'field_5c0df98063410',
			'label' => __('URL', 'blogman'),
			'name' => '_format_link_url',
			'type' => 'text',
			'instructions' => '',
			'required' => 0,
			'conditional_logic' => 0,
			'wrapper' => array(
				'width' => '',
				'class' => '',
				'id' => '',
			),
			'default_value' => '',
			'placeholder' => '',
			'prepend' => '',
			'append' => '',
			'maxlength' => '',
		),
	),
	'location' => array(
		array(
			array(
				'param' => 'post_format',
				'operator' => '==',
				'value' => 'link',
			),
		),
	),
	'menu_order' => 0,
	'position' => 'normal',
	'style' => 'default',
	'label_placement' => 'top',
	'instruction_placement' => 'label',
	'hide_on_screen' => '',
	'active' => 1,
	'description' => '',
));

acf_add_local_field_group(array(
	'key' => 'group_5c0df4a18fef1',
	'title' => __('Enter Video Detail', 'blogman'),
	'fields' => array(
		array(
			'key' => 'field_5c0df4b8d03ad',
			'label' => __('VIDEO URL (OEMBED) OR EMBED CODE', 'blogman'),
			'name' => '_format_video_embed',
			'type' => 'textarea',
			'instructions' => '',
			'required' => 0,
			'conditional_logic' => 0,
			'wrapper' => array(
				'width' => '',
				'class' => '',
				'id' => '',
			),
			'default_value' => '',
			'placeholder' => '',
			'maxlength' => '',
			'rows' => 4,
			'new_lines' => '',
		),
	),
	'location' => array(
		array(
			array(
				'param' => 'post_format',
				'operator' => '==',
				'value' => 'video',
			),
		),
	),
	'menu_order' => 0,
	'position' => 'normal',
	'style' => 'default',
	'label_placement' => 'top',
	'instruction_placement' => 'label',
	'hide_on_screen' => '',
	'active' => 1,
	'description' => '',
));

endif;
 

/**
 * One Click Demo Import
 */
function blogman_ocdi_import_files() {
  return array(
    array(
      'import_file_name'           => 'Demo',
      'import_file_url'            => 'http://demo.awaikenthemes.com/blogman/dummy-data/blogman-post.xml',
      'import_widget_file_url'     => 'http://demo.awaikenthemes.com/blogman/dummy-data/blogman-widgets.wie',
      'import_customizer_file_url' => 'http://demo.awaikenthemes.com/blogman/dummy-data/blogman-customize.dat',
      'preview_url'                => 'http://demo.awaikenthemes.com/blogman/',
    ),
  );
}
add_filter( 'pt-ocdi/import_files', 'blogman_ocdi_import_files' );


function blogman_ocdi_after_import_setup() {
	// Assign menus to their locations.
	$main_menu = get_term_by( 'name', 'Main Menu', 'nav_menu' );

	set_theme_mod( 'nav_menu_locations', array(
			'main-menu' => $main_menu->term_id,
		)
	);
}
add_action( 'pt-ocdi/after_import', 'blogman_ocdi_after_import_setup' );

add_filter( 'pt-ocdi/disable_pt_branding', '__return_true' );